#!/usr/bin/python

import os
import sys
import time
import serial
from serial import SerialException

serial_port = os.getenv("SERIAL_PORT") or "/dev/ttyS2"
baudrate = os.getenv("BAUD_RATE") or "115200"
serial_timeout = int(os.getenv("SERIAL_TIMEOUT") or "1")

# GPIO Definitions
GPIO_BASE_PATH = '/sys/devices/platform/gpio/'
GPIO_CARD_NRESET = 'card_nreset'
GPIO_BT_BOOT_MODE = 'bt_boot_mode'

# BT boot modes
BT_BOOTLOADER_MODE = 0
BT_APP_MODE = 1

def set_gpio_value(gpio_name, value):
    with open(GPIO_BASE_PATH + gpio_name + '/value', 'w') as f:
        f.write('%d' % int(value))

def set_bt_boot_mode(mode):
    print(f"Setting BL654 boot mode to {mode}")
    set_gpio_value(GPIO_BT_BOOT_MODE, mode)
    set_gpio_value(GPIO_CARD_NRESET, 0)
    set_gpio_value(GPIO_CARD_NRESET, 1)
    time.sleep(2)

def send_raw(bytes_to_send, rx_timeout=1):
    ser.reset_input_buffer()
    ser.timeout = rx_timeout
    ser.write(bytes_to_send)
    resp = ser.read(1024)
    print(f"Raw serial response bytes: {resp}")
    return resp

def send_bytes(bytes_to_send, rx_timeout=1):
    try:
        resp_bytes = send_raw(bytes_to_send, rx_timeout)
        read_stuff = resp_bytes.decode('utf-8').strip().replace('\t', ',').split('\r\n')
        print(f'Retrieved read_stuff as {read_stuff}')
    except UnicodeDecodeError:
        read_stuff = 'Caught exception trying to decode something from serial port'
        print(read_stuff)
    return read_stuff

def get_version():
    resp_bytes = send_raw(b'i2\x01')
    print(f"App version raw response: {resp_bytes}")
    # "i2\x01\x00\x00"
    if len(resp_bytes) >= 5:
        app_ver = resp_bytes[3] + (256 * resp_bytes[4])
    print(f"App version: {app_ver}")

def get_pubkey():
    resp_bytes = send_raw(b'y2')
    print(f"Pubkey raw response: {resp_bytes}")

def erase_btpa():
    # send platform selection command
    plat_resp = send_raw(b'p\x4F\x65\xCB\x5E')
    if plat_resp == b"a":
        print(f"Received correct platform response: {plat_resp}, proceeding with full erase")
    else:
        print("Did not receive correct platform response, aborted erase attempt")
        return
    # send erase all command & await response up to 20 seconds
    erase_resp = send_raw(b"\x7F\x7F", 20)
    print(f"Received erase response: {erase_resp}")
    # send the application version query
    app_chk_resp = send_raw(b'i2\x01')
    if app_chk_resp == b"i2\x01\x00\x00":
        print("app_chk_resp checks out!")
    else:
        print("app_chk_resp not correct!")
        return
    # send the public key query
    pubkey_resp = send_raw(b"y2")
    if pubkey_resp == b"y2\x00":
        print("pubkey_resp checks out!")
    else:
        print("pubkey_resp not correct!")
        return

def usage():
    print("Usage: bl654_bt_util.py [version|erase]")
    sys.exit(1)

# check arguments
if len(sys.argv) < 2:
    usage()
action = sys.argv[1]
if action != "version" and action != "erase" and action != "pubkey":
    usage()

# Reset into bootloader mode
set_bt_boot_mode(BT_BOOTLOADER_MODE)

# open serial port
try:
    # Open the COM port to the Bluetooth adapter
    ser = serial.Serial(serial_port, int(baudrate), timeout=serial_timeout)
    ser.reset_input_buffer()
    print(f"Successfully opened serial port {serial_port}@{baudrate}")
except SerialException:
    print(f"Unable to open port: {serial_port}")
    sys.exit(1)

# "M" is the bootloader version query command
bootchk_resp = send_raw(b"M")
if "Bootloader version" in bootchk_resp.decode('utf-8'):
    print(f"Successfully entered bootloader, version: {bootchk_resp}")
else:
    print("Could not verify bootloader was running")
    sys.exit(1)

if action == "version":
    get_version()
elif action == "pubkey":
    get_pubkey()
elif action == "erase":
    erase_btpa()

# Reset to app mode
set_bt_boot_mode(BT_APP_MODE)
